//
//  AppleScriptRunner.swift
//

import Cocoa
import Carbon

extension FileManager {
    var applicationScriptsURL : URL {
        get async { try! url(for: .applicationScriptsDirectory, in: .userDomainMask, appropriateFor: nil, create: true) }
    }
}

fileprivate enum EventKey {
    static let typePSN = UInt32(typeProcessSerialNumber)
    static let currentProcess = UInt32(kCurrentProcess)
    static let appleScriptSuite = FourCharCode("ascr")
    static let subroutineEvent = FourCharCode("psbr")
    static let autoGenerateReturnID = Int16(kAutoGenerateReturnID)
    static let anyTransactionID = Int32(kAnyTransactionID)
    static let subroutineNameKey = FourCharCode("snam")
    static let directObjectKey = UInt32(keyDirectObject)
    static let asUserRecordFields = UInt32(keyASUserRecordFields)
    static let missingValue = FourCharCode("msng")
}

actor AppleScriptRunner {
    
    enum Handler : String {
        case finderSelection
        
        var fileName : String { return "ApplicationScriptsTest" }
    }
    
    enum RunnerError : Error {
        case bundleNameNotSpecified
        case bundleIdentifierNotSpecified
        case scriptFileMissing
        case execution(String)
        case typeMismatch(String)
        
        var localizedDescription : String {
            switch self {
                case .bundleNameNotSpecified: return "A bundle name is not specified"
                case .bundleIdentifierNotSpecified: return "A bundle identifier is not specified"
                case .scriptFileMissing: return "The script file is missing in the bundle"
                case .execution(let message): return message
                case .typeMismatch(let message): return message
            }
        }
    }
        
    func run<T : Sendable>(handler: Handler, fileName: String? = nil, parameters: [DescriptorConvertible] = []) async throws -> T {
        let event = eventFor(handler: handler, inputParameters: parameters.map{ $0.scriptingDescriptor })
        let resultDescriptor = try await runScript(fileName: fileName ?? handler.fileName, event: event)
        guard let result = resultDescriptor.objectValue as? T else {
            throw RunnerError.typeMismatch("Type mismatch descriptor type: '\(resultDescriptor.descriptorType.stringRepresentation)'")
        }
        return result
    }
    
    func run(handler: Handler, fileName: String? = nil, parameters: [DescriptorConvertible] = []) async throws {
        let event = eventFor(handler: handler, inputParameters: parameters.map{ $0.scriptingDescriptor })
        try await runScript(fileName: fileName ?? handler.fileName, event: event)
    }
    
    @discardableResult
    private func runScript(fileName: String, event: NSAppleEventDescriptor? = nil) async throws -> NSAppleEventDescriptor {
        
        let scriptFolderURL = try FileManager.default.url(for: .applicationScriptsDirectory, in: .userDomainMask, appropriateFor: nil, create: false)
        let documentsPathScript = scriptFolderURL.appendingPathComponent(fileName).appendingPathExtension("scpt")
        do {
            let script = try NSUserAppleScriptTask(url: documentsPathScript)
            return try await withCheckedThrowingContinuation({ continuation in // Continuation API used to avoid Sendable error
                script.execute(withAppleEvent: event) { descriptor, error in
                    if let error { continuation.resume(throwing: error) } else {
                        continuation.resume(returning: descriptor!)
                    }
                }
            })
        } catch {
            guard let failureReason = (error as NSError).userInfo["NSLocalizedFailureReason"] as? String,
                  let scriptRange = failureReason.range(of: #"/\w+\.scpt.*"#, options: .regularExpression) else {
                throw error
            }
            throw RunnerError.execution(String(failureReason[scriptRange].dropFirst()))
        }
        
    }
    
    private func eventFor(handler: Handler, inputParameters parameters: [NSAppleEventDescriptor]) -> NSAppleEventDescriptor
    {
        var processSerialNumber = ProcessSerialNumber(highLongOfPSN: 0, lowLongOfPSN: EventKey.currentProcess)
        
        let target = NSAppleEventDescriptor(descriptorType: EventKey.typePSN,
                                            bytes: &processSerialNumber,
                                            length: MemoryLayout<ProcessSerialNumber>.size)
        let function = NSAppleEventDescriptor(string: handler.rawValue)
        let event = NSAppleEventDescriptor(eventClass: EventKey.appleScriptSuite,
                                           eventID: EventKey.subroutineEvent,
                                           targetDescriptor: target,
                                           returnID: EventKey.autoGenerateReturnID,
                                           transactionID: EventKey.anyTransactionID)
        event.setParam(function, forKeyword: EventKey.subroutineNameKey)
        if !parameters.isEmpty {
            let parameterList = NSAppleEventDescriptor.list()
            parameters.forEach { parameterList.insert($0, at: 0) } // appends the item
            event.setParam(parameterList, forKeyword: EventKey.directObjectKey)
        }
        return event
    }
    
    @MainActor
    func installScript() async throws {
        let fileManager = FileManager.default
        let bundle = Bundle.main
       
        guard let appName = bundle.infoDictionary?["CFBundleName"] as? String else { throw RunnerError.bundleNameNotSpecified }
        let directoryURL = await fileManager.applicationScriptsURL
        let scriptName = appName + ".scpt"
        if fileManager.fileExists(atPath: directoryURL.appendingPathComponent(scriptName).path) { return }
        guard let bundleID = bundle.bundleIdentifier else { throw RunnerError.bundleIdentifierNotSpecified }
        guard let sourceURL = Bundle.main.url(forResource: appName, withExtension: "scpt") else { throw RunnerError.scriptFileMissing }
        let panel = NSOpenPanel()
        panel.directoryURL = directoryURL
        panel.canChooseDirectories = true
        panel.canChooseFiles = false
        panel.prompt = "Select Script Folder"
        panel.message = "Please select the User > Library > Application Scripts > \(bundleID) folder"
        let response = await panel.begin()
        if response == .OK, let url = panel.url, url == directoryURL {
            try fileManager.copyItem(at: sourceURL, to: directoryURL.appendingPathComponent(scriptName))
            let alert = NSAlert()
            alert.addButton(withTitle: NSLocalizedString("OK", comment:""))
            alert.messageText =  "Helper Script Installed"
            alert.informativeText = "The helper script \(scriptName) was installed succcessfully."
            alert.runModal()
        } else {
            // wrong directory or user cancelled, try again
            try await self.installScript()
        }
    }
}

//MARK: - Scripting Conversions

protocol DescriptorConvertible : Sendable {
    var scriptingDescriptor : NSAppleEventDescriptor { get }
}

extension Array : DescriptorConvertible where Element : DescriptorConvertible {
  
    var scriptingDescriptor : NSAppleEventDescriptor {
        let listDesc = NSAppleEventDescriptor.list()
        self.forEach { listDesc.insert($0.scriptingDescriptor, at: 0) }
        return listDesc
    }
}

extension Dictionary : DescriptorConvertible where Key == String, Value: DescriptorConvertible {
    
    var scriptingDescriptor : NSAppleEventDescriptor {
        let recordDesc = NSAppleEventDescriptor.record()
        let userFieldDesc = NSAppleEventDescriptor.list()
        
        for (key, value) in self {
            userFieldDesc.insert(key.scriptingDescriptor, at: 0)
            userFieldDesc.insert(value.scriptingDescriptor, at: 0)
        }
        recordDesc.setDescriptor(userFieldDesc, forKeyword: EventKey.asUserRecordFields)
        return recordDesc
    }
}

extension String : DescriptorConvertible {
    var scriptingDescriptor : NSAppleEventDescriptor { NSAppleEventDescriptor(string: self) }
}

extension URL : DescriptorConvertible {
    var scriptingDescriptor : NSAppleEventDescriptor { NSAppleEventDescriptor(fileURL: self) }
}

extension Int : DescriptorConvertible {
    var scriptingDescriptor :  NSAppleEventDescriptor { NSAppleEventDescriptor(int32: Int32(self)) }
}

extension Int32 : DescriptorConvertible {
    var scriptingDescriptor :  NSAppleEventDescriptor { NSAppleEventDescriptor(int32: self) }
}

extension Bool : DescriptorConvertible {
    var scriptingDescriptor :  NSAppleEventDescriptor { NSAppleEventDescriptor(boolean: self) }
}

extension Double : DescriptorConvertible {
    var scriptingDescriptor : NSAppleEventDescriptor { NSAppleEventDescriptor(double: self) }
}

extension Date : DescriptorConvertible {
    var scriptingDescriptor : NSAppleEventDescriptor { NSAppleEventDescriptor(date: self) }
}

extension Data : DescriptorConvertible {
    var scriptingDescriptor : NSAppleEventDescriptor { NSAppleEventDescriptor(descriptorType: typeData, data: self)! }
}

extension NSAppleEventDescriptor {
    
    var objectValue : Any {
        let descType = descriptorType
        var object : Any
        //print(descType)
        //print(descType.stringRepresentation)
        
        switch descType {
            case typeUnicodeText, typeUTF8Text: object = stringValue!
            case typeFileURL: object = fileURLValue!
            case typeAlias: object = self.coerce(toDescriptorType: typeFileURL)!.fileURLValue!
            case typeAEList: object = scriptingList()
            case typeAERecord: object = scriptingRecord()
            case typeTrue: object = true
            case typeFalse: object = false
            case typeBoolean: object = booleanValue
            case typeSInt16, typeUInt16,
                typeSInt32, typeUInt32,
                typeSInt64, typeUInt64:
                object = Int(int32Value)
            case typeIEEE32BitFloatingPoint, typeIEEE64BitFloatingPoint: object = doubleValue
            case typeLongDateTime: object = dateValue!
            case typeData: object = data
            case typeType where typeCodeValue == EventKey.missingValue: object = NSNull()
                
            default: object = withUnsafeBytes(of: Int(descType)) { Data($0) }
        }
        return object
    }
    
    func scriptingList() -> [Any] {
        return (1...numberOfItems).compactMap { itemIndex in
            return self.atIndex(itemIndex)?.objectValue
        }
    }
    
    func scriptingRecord() -> [String:Any] {
        
        var dict = [String:Any]()
        guard let fieldItems = self.forKeyword(EventKey.asUserRecordFields) else { return dict }
        let numItems = fieldItems.numberOfItems
        
        for itemIndex in stride(from: 1, to: numItems, by: 2) {
            let keyDesc = fieldItems.atIndex(itemIndex)!
            let valueDesc = fieldItems.atIndex(itemIndex + 1)!
            if let keyString = keyDesc.stringValue {
                dict[keyString] = valueDesc.objectValue
            }
        }
        return dict
    }
}

private extension FourCharCode {
    init(_ string : String) {
        self = string.utf16.reduce(0, {$0 << 8 + FourCharCode($1)})
    }
}

private extension UInt32 {
    var stringRepresentation : String {
        let data = withUnsafeBytes(of: self.bigEndian) { Data($0) }
        return String(data: data, encoding: .macOSRoman)!
    }
}
